% Materia wprowadzajcy do zagadnienia klasteryzacji danych. Materia
% obejmuje funkcje zawarte w Matlab Statistic Toolbox, dotyczce metod:
% klasteryzacji hierarchicznej; klasteryzacji typu k-means, klasteryzacji
% typu: Gaussian Mixture oraz Soft Clustering Using Gaussian Mixture Distributions.

% Opracowa dr in. Micha Grchowski, na podstawie Matlab Statistic Toolbox.

%%

% *************************************************************************
% *********     klasteryzacja hierarchiczna "step by step"    *************
% *************************************************************************

clc
close all

% przykadowe dane
X = [1 2;2.5 4.5;2 2;4 1.5;4 2.5]
plot(X(:,1), X(:,2),'.')
axis([0 5 0 5])
grid on

Y = pdist(X, 'euclidean')
            % lub np.Y1 = pdist(X,'cityblock')

% funkcja pdist pozwala na obliczanie dystansu pomidzy poszczeglnymi punktami 
% zbioru X na podstawie przyjtej metryki. Szczegy funkcji dostpne w  
% doc help pdist. 
% Za pomoc funkcji squareform moliwe jest utworzenie macierzy kwadratowej 
% zawierajcej te odlegoci

dist_matrix1=squareform(Y)
            % lub np. dist_matrix1=squareform(Y1)

% w powyszym przykadzie element 1,1 jest odlegoci elementu pierwszego 
% od pierwszego, 1,2  - jest odlegoci elementu pierwszego od drugiego itd... 

% funkcja linkage korzystajc z informacji wygenerowanej przez pdist, czy
% ze sob elementy lece "najbliej" siebie w clustry (2 elementowe).
% Nastpnie docza kolejne elementy itd a zostanie utworzone cae
% hierarchiczne drzewo zawierajce wsyztskie elementy.

Z = linkage(Y)

% kady z wierszy Z okrela poczenie elementw lub klastrw. Pierwsze 2
% kolumny okrelaj poczone obiekty, a trzecia kolumna okrela odlego
% pomiedzy nimi. 
% w powyszym przykadzie w 3 wierszu pojawia si poczenie elementu 6 i
% 7 (dane orginalne zawieray jedynie 5 danych). Element 6 jest klastem
% zawierajcym orginalne elementy 4 i 5, natomiast element 7 jest klastem
% zawierajcym orginalne elementy 1 i 3 .Element 8 to zgrupowane dane 1,3,4,5
% informacje zawarte w linkage mona zwizualizowac w postaci grafu (drzewa)
% przy pomocy funkcji dendrogram

figure;
dendrogram(Z)

% o pozioma okrela indeksy orginalnych danych. Wysoko drzewa okrelona 
% jest poprzez odlegoci pomidzy klastrami. Np. link reprezentujcy klaster 
% zawierajcy element 1 i 3 ma wysoko 1. link reprezentujcy klaster 
% grupujcy element 2 z 8 ma wysoko 2.5 (jest to wysoko pomidzy elementami 2 i 8)

% elementy skadowe klastrw powinny by jak najbardziej ze sob
% skorelowane. Do analizy korelacji tych suy funkcja cophenet. Im blisze 
% jej wyjcie jest wartoci 1 tym lepiej.

c  = cophenet(Z,Y)

%  kolejna z funkcji inconsistent pozwala na obliczanie "odlegoci" 
%  (rnica wysokoci klastra  i redni wysokoci klastra, normalizowana 
%  przez odchylenie standardowe).  

I  = inconsistent(Z)

% Tworzenie klastrw 
% *************************************************************************
% 
% Klastry mona utworzy na dwa sposoby:
% 1. Na podstawie ustalanie poziomu miary inconsistent, poprzez "przecicie" 
% drzewa w miejscu gdy miara ta przekroczy zaoony podzia: 

T  = cluster(Z,'cutoff',1.2)

% lub np
%T  = cluster(Z,'cutoff',0.8)

% wyjscie z funkcji okrela ktry obiekt zosta przydzielony do ktrego
% klastra.

% 2. Na podstawie zaoonej apriori liczby klastrw

T  = cluster(Z,'maxclust',2)
% lub np
%T  = cluster(Z,'maxclust',3)

% Na koniec mona wyrysowa dane po podziale, np tak: 

figure;                
plot(X(T==1,1),X(T==1,2),'r.','MarkerSize',20)
hold on
plot(X(T==2,1),X(T==2,2),'b.','MarkerSize',20)
hold on
plot(X(T==3,1),X(T==3,2),'g.','MarkerSize',20)

legend('Cluster 1','Cluster 2','Cluster 3','Location','NW')
%%

% *************************************************************************
% ************            klasteryzacja typu k-means    *******************
% *************************************************************************

close all;
clear;
clc

% przykadowe dane 
load kmeansdata;
size(X);

% funkcja kmeans dzieli dane na z gry okrelon ilo klustrw posugujc si 
% okrelon metryk odlegoci, np:  

idx3 = kmeans(X,3,'distance','city');

% aby sprawdzi czy ten podzia jest adekwatny, mona uy funkcji silhouette.
% funkcja ta zwraca warto 1 gdy punkt jest "idealnie wpasowany w dany klaster, 
% 0 gdy tak samo pasuje do tego jak i innego klastra oraz 1 gdy na pewno nie 
% naley do tego klastra (wartoci pomidzy -1 a 1 niuansuj powysze).

figure;
[silh3,h] = silhouette(X,idx3,'city','display','iter');
set(get(gca,'Children'),'FaceColor',[.8 .8 1])
xlabel('Silhouette Value')
ylabel('Cluster')

% zwiksz i zmniejsz liczb klastrw aby zobaczy ktra z nich jest
% najlepsza. 
% Uyj funkcji mean aby obliczy precyzyjnie redni warto silhouette 

mean(silh3)

% parametr kmeans 'replicates' pozwala na ponawianie iteracyjne algorytmu w
% celu uniknicia wpadania w lokalne minima. 

[idx4,centroids,sumdist] = kmeans(X,4,'dist','city',...
'display','final','replicates',5);

sum(sumdist)

% dane pogrupowane s do zbiorw, np:
% X(idx4==1); X(idx4==2); X(idx4==3); X(idx4==4);
% X_klaster1=X(idx4==1,1:size(X,2))
% X_klaster1=X(idx4==2,1:size(X,2))
% X_klaster1=X(idx4==3,1:size(X,2))
% X_klaster1=X(idx4==4,1:size(X,2))

% ostatecznie mona wyrysowa klastry

figure;                
plot(X(idx4==1,1),X(idx4==1,2),'r.','MarkerSize',12)
hold on
plot(X(idx4==2,1),X(idx4==2,2),'b.','MarkerSize',12)
hold on
plot(X(idx4==3,1),X(idx4==3,2),'g.','MarkerSize',12)
hold on
plot(X(idx4==4,1),X(idx4==4,2),'y.','MarkerSize',12)
plot(centroids(:,1),centroids(:,2),'kx',...
      'MarkerSize',12,'LineWidth',2)
plot(centroids(:,1),centroids(:,2),'ko',...
      'MarkerSize',12,'LineWidth',2)
legend('Cluster 1','Cluster 2','Cluster 3','Cluster 4','Centroids',...
      'Location','NW')

% zwr uwag i dane orginalne (i pogrupowane) s wymiaru nx4 wic
% powinnimy obejrze rwnie inne wymiary np:

figure;                
plot(X(idx4==1,1),X(idx4==1,4),'r.','MarkerSize',12)
hold on
plot(X(idx4==2,1),X(idx4==2,4),'b.','MarkerSize',12)
hold on
plot(X(idx4==3,1),X(idx4==3,4),'g.','MarkerSize',12)
hold on
plot(X(idx4==4,1),X(idx4==4,4),'y.','MarkerSize',12)
plot(centroids(:,1),centroids(:,4),'kx',...
      'MarkerSize',12,'LineWidth',2)
plot(centroids(:,1),centroids(:,4),'ko',...
      'MarkerSize',12,'LineWidth',2)
legend('Cluster 1','Cluster 2','Cluster 3','Cluster 4','Centroids',...
      'Location','NW')


%%


% *************************************************************************
% ************      klasteryzacja typu Gaussian Mixture Models    *********
% *************************************************************************

% modele tego typu lepiej si sprawdzaj od k-means clustering w
% przypadkach gdy dane w klastrach maj rzny rozmiar i korelacj wzajemn.
% Algorytmy tego typu czsto nazywane s "mikkimi" poniewa umoliwiaj
% naleenie do kilku klastrw z pewnym stopniem przynaleznoci. 

close all;
clear;
clc

% przykadowe dane 
mu1 = [1 2];
sigma1 = [3 .2; .2 2];
mu2 = [-1 -2];
sigma2 = [2 0; 0 1];
X = [mvnrnd(mu1,sigma1,200);mvnrnd(mu2,sigma2,100)];

figure;
scatter(X(:,1),X(:,2),10,'ko')

% Funkcja gmdistribution.fit "wpasowuje" Gaussian Mixture Model w dane. 
% W tym przypadku uylimy 2 klastrw 

options = statset('Display','final');
gm = gmdistribution.fit(X,2,'Options',options)

% funkcja ezcontour umoliwia wykrelenie wykresu dla utworzonego modelu.

hold on
ezcontour(@(x,y)pdf(gm,[x y]),[-8 6],[-8 6]);
hold off

% funkcja cluster umoliwia przypisanie danych do obu klastrw
% wykorzystujc gmdistribution.fit

figure;
idx = cluster(gm,X);
cluster1 = (idx == 1);
cluster2 = (idx == 2);
scatter(X(cluster1,1),X(cluster1,2),10,'r+');
hold on
scatter(X(cluster2,1),X(cluster2,2),10,'bo');
hold off
legend('Cluster 1','Cluster 2','Location','NW')

% wiemy jednak e punkty nie nale do danych klastrw z rwnym stopniem
% przynalenoci. Informacj t rwniez mozemy wykorzysta przy pomocy
% funkcji posterior

figure;
P = posterior(gm,X);
scatter(X(cluster1,1),X(cluster1,2),10,P(cluster1,1),'+')
hold on
scatter(X(cluster2,1),X(cluster2,2),10,P(cluster2,1),'o')
hold off
legend('Cluster 1','Cluster 2','Location','NW')
clrmap = jet(80); colormap(clrmap(9:72,:))
ylabel(colorbar,'Component 1 Posterior Probability')
 
%%
% *************************************************************************
% ** mikka klasteryzacja wykorzystujca Gaussian Mixture Distributions ***
% *************************************************************************

% wykorzystujc funkcj posterior, punkty mog zosta posortowane wzgldem
% stopni przynalenoci do klastrw i przedtsawione w nastpujcy sposb

figure;
[~,order] = sort(P(:,1));
plot(1:size(X,1),P(order,1),'r-',1:size(X,1),P(order,2),'b-');
legend({'Cluster 1 Score' 'Cluster 2 Score'},'location','NW');
ylabel('Cluster Membership Score');
xlabel('Point Ranking');

% jak wida na tym wykresie separacja jest w miar dokadna (ostra).
% Niewiele punktw ma np stopie przynaleznoci 0.5. 

% Gaussian Mixture waona macierz covariancji jest bardzo zbliony do 
% fuzzy c - means clustering

gm2 = gmdistribution.fit(X,2,'CovType','Diagonal',...
'SharedCov',true)

figure;
P2 = posterior(gm2,X); 
% ewentualnie: [idx,P2] = cluster(gm2,X)
[~,order] = sort(P2(:,1));
plot(1:size(X,1),P2(order,1),'r-',1:size(X,1),P2(order,2),'b-');
legend({'Cluster 1 Score' 'Cluster 2 Score'},'location','NW');
ylabel('Cluster Membership Score');
xlabel('Point Ranking');


% *******   Przypisywanie nowych danych do utworzonych klastrw   *********

% nowe dane (Y) mozemy przypisa do istniejacych klastrw w nastpujcy
% sposb

figure
Y = [mvnrnd(mu1,sigma1,50);mvnrnd(mu2,sigma2,25)];
idx = cluster(gm,Y);
cluster1 = (idx == 1);
cluster2 = (idx == 2);
scatter(Y(cluster1,1),Y(cluster1,2),10,'r+');
hold on
scatter(Y(cluster2,1),Y(cluster2,2),10,'bo');
hold off
legend('Class 1','Class 2','Location','NW')



